
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   UTIL.C - Utility Functions
// 
//
//   DESCRIPTION
//
//   Functions used to retrieve and put common bit and integer data types,
//   calculate 16-bit checksums, and convert lat/lon from decimal degrees to
//   degrees and minutes.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\UTIL.C_V  $
//   
//      Rev 1.15   09 Nov 1998 10:48:12   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.5   Jul 09 1997 10:57:32   COLEJ
//    
//   
//      Rev 1.4   Feb 25 1997 14:39:54   COLEJ
//   added ENU matrix for delta calculations
//   
//      Rev 1.3   Feb 12 1997 16:05:56   COLEJ
//    
//   
//      Rev 1.2   Oct 31 1996 11:29:34   COLEJ
//    
//   
//      Rev 1.1   Aug 13 1996 15:11:26   COLEJ
//    
//   
//      Rev 1.0   13 May 1996 14:53:24   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////
            
#include <math.h>
#include <stdio.h>
#include <conio.h>
#include <graph.h>

#include "gentypes.h"
#include "gencons.H"
#include "ztypes.h"   
#include "util.h"   


////////////////////////////////////////////////////////////////////////
//
//   Get a bit variable type from a ZMsgBuf of 16 bit binary data.
//
void GetBit(tBOOL *data, tMSGBUF *ZMsgBuf, short i, short j)
{
   *data = (*(tBOOL *) (&ZMsgBuf->word[i-1]) >> j) & 0x01;
}
  
////////////////////////////////////////////////////////////////////////
//
//   Get a group of bits from a ZMsgBuf of 16 bit binary data.
//
void GetBits(tUSHORT *data, tMSGBUF *ZMsgBuf, short i, short j, short k)
{  
   tUSHORT Mask;
   
   Mask = 0xFFFF >> (16-k);
   *data = (*(tUSHORT *) (&ZMsgBuf->word[i-1]) >> j) & Mask;
}

////////////////////////////////////////////////////////////////////////
//
//   Get a group of bits as char from a ZMsgBuf of 16 bit binary data.
//
void GetChar(tCHAR *data, tMSGBUF *ZMsgBuf, short i, short j)
{  
   *data = (tCHAR) (ZMsgBuf->word[i-1] >> j*8 & 0x00FF);
}

////////////////////////////////////////////////////////////////////////
//
//   Get a boolean variable type from a ZMsgBuf of 16 bit binary data.
//
void GetBool(tBOOL *data, tMSGBUF *ZMsgBuf, short i)
{   
   *data = *(tBOOL *) (&ZMsgBuf->word[i-1]);
}  

////////////////////////////////////////////////////////////////////////
//
//   Get a short variable type from a ZMsgBuf of 16 bit binary data.
//
void GetShort(tSHORT *data, tMSGBUF *ZMsgBuf, short i)
{   
   *data = *(tSHORT *) (&ZMsgBuf->word[i-1]);
}  

////////////////////////////////////////////////////////////////////////
//
//   Get a long variable type from a ZMsgBuf of 16 bit binary data.
//
void GetLong(tLONG *data, tMSGBUF *ZMsgBuf, short i)
{
   *data = *(tLONG *)(&ZMsgBuf->word[i-1]);
} 

////////////////////////////////////////////////////////////////////////
//
//   Get an unsigned short variable type from a ZMsgBuf 
//   of 16 bit binary data.
//
void GetUShort(tUSHORT *data, tMSGBUF *ZMsgBuf, short i)
{
   *data = *(tUSHORT *)(&ZMsgBuf->word[i-1]);
}  

////////////////////////////////////////////////////////////////////////
//
//   Get an unsigned long variable type from a ZMsgBuf of 16 bit binary data.
//
void GetULong(tULONG *data, tMSGBUF *ZMsgBuf, short i)
{
   *data = *(tULONG *)(&ZMsgBuf->word[i-1]);
}

////////////////////////////////////////////////////////////////////////
//
//   Get a triple variable type from a ZMsgBuf of 16 bit binary data.
//
void GetTriple(tTRIPLE *data, tMSGBUF *ZMsgBuf, short i)
{
   data->data[0] = *(tUSHORT *)(&ZMsgBuf->word[i-1]);
   data->data[1] = *(tUSHORT *)(&ZMsgBuf->word[i]  );
   data->data_2  = *(tSHORT  *)(&ZMsgBuf->word[i+1]);
}

////////////////////////////////////////////////////////////////////////
//
//   Get a unsigned triple variable type from a ZMsgBuf of 16 bit binary data.
//
void GetUTriple(tUTRIPLE *data, tMSGBUF *ZMsgBuf, short i)
{
   data->data[0] = *(tUSHORT *)(&ZMsgBuf->word[i-1]);
   data->data[1] = *(tUSHORT *)(&ZMsgBuf->word[i]  );
   data->data[2] = *(tUSHORT *)(&ZMsgBuf->word[i+1]);
}

////////////////////////////////////////////////////////////////////////
//
//   Get a float variable type from a ZMsgBuf of 16 bit binary data.
//
void GetFloat(tFLOAT *data, tMSGBUF *ZMsgBuf, short i)
{
   *data = *(tFLOAT *)(&ZMsgBuf->word[i-1]);
}  

////////////////////////////////////////////////////////////////////////
//
//   Get a double variable type from a ZMsgBuf of 16 bit binary data.
//
void GetDouble(tDOUBLE *data, tMSGBUF *ZMsgBuf, short i)
{
   *data = *(tDOUBLE *)(&ZMsgBuf->word[i-1]);
}

////////////////////////////////////////////////////////////////////////
//
//   Put a bit variable type into a ZMsgBuf of 16 bit binary data.
//
void PutBit(tBOOL word, tMSGBUF *ZMsgBuf, short i, short j)
{
   *(&ZMsgBuf->word[i-1]) = *(&ZMsgBuf->word[i-1]) | ((word & 0x01) << j);
}

////////////////////////////////////////////////////////////////////////
//
//   Put a group of bits into a ZMsgBuf of 16 bit binary data.
//
void PutBits(tUSHORT word, tMSGBUF *ZMsgBuf, short i, short j, short k)
{  
   tUSHORT Mask;
   
   Mask = 0xFFFF >> (16-k);
   *(&ZMsgBuf->word[i-1]) = *(&ZMsgBuf->word[i-1]) | ((word & Mask) << j);
}

////////////////////////////////////////////////////////////////////////
//
//   Put a group of bits as char into a ZMsgBuf of 16 bit binary data.
//
void PutChar(tCHAR word, tMSGBUF *ZMsgBuf, short i, short j)
{
   *(&ZMsgBuf->word[i-1]) = *(&ZMsgBuf->word[i-1])|(word&(0x00FF<<j*8));
}  

////////////////////////////////////////////////////////////////////////
//
//   Put a boolean variable type into a ZMsgBuf of 16 bit binary data.
//
void PutBool(tBOOL word, tMSGBUF *ZMsgBuf, short i)
{
   *(tBOOL *)(&ZMsgBuf->word[i-1]) = word;   
}

////////////////////////////////////////////////////////////////////////
//
//   Put a short variable type into a ZMsgBuf of 16 bit binary data.
//
void PutShort(tSHORT word, tMSGBUF *ZMsgBuf, short i)
{
   *(tSHORT *)(&ZMsgBuf->word[i-1]) = word;   
}  

////////////////////////////////////////////////////////////////////////
//
//   Put a long variable type into a ZMsgBuf of 16 bit binary data.
//
void PutLong(tLONG word, tMSGBUF *ZMsgBuf, short i)
{
   *(tLONG *)(&ZMsgBuf->word[i-1]) = word;
}

////////////////////////////////////////////////////////////////////////
//
//   Put an unsigned short variable type into a ZMsgBuf of 16 bit binary data.
//
void PutUShort(tUSHORT word, tMSGBUF *ZMsgBuf, short i)
{
   *(tUSHORT *)(&ZMsgBuf->word[i-1]) = word;   
}  

////////////////////////////////////////////////////////////////////////
//
//   Put an unsigned long variable type into a ZMsgBuf of 16 bit binary data.
//
void PutULong(tULONG word, tMSGBUF *ZMsgBuf, short i)
{
   *(tULONG *)(&ZMsgBuf->word[i-1]) = word;
}

////////////////////////////////////////////////////////////////////////
//
//   Put a float variable type into a ZMsgBuf of 16 bit binary data.
//
void PutFloat(tFLOAT word, tMSGBUF *ZMsgBuf, short i)
{
   *(tFLOAT *)(&ZMsgBuf->word[i-1]) = word;
}  

////////////////////////////////////////////////////////////////////////
//
//   Put a double variable type into a ZMsgBuf of 16 bit binary data.
//
void PutDouble(tDOUBLE word, tMSGBUF *ZMsgBuf, short i)
{
   *(tDOUBLE *)(&ZMsgBuf->word[i-1]) = word;
}

///////////////////////////////////////////////////////////////////
// 
//  Calculate a 16 bit check sum.
//
short CheckSum(tMSGBUF *ZMsgBuf, tSHORT msglen)
{
   short i, len = 0;
   long cs;

   // Calculate the check sum 
   cs = 0L;   
   len = (len - 1) * 2;
  
   // Calculate the check sum of the message data
   for(i=0 ;i<=len; i=i+2){
      cs = cs + *(short*)(&ZMsgBuf->word[i-1]);
   }
   cs = -cs;
  
   return((short)(cs & 0xFFFF));  
}

////////////////////////////////////////////////////////////////////////
//
//   Show a text string at a postion
//
void ShowText(char *buffer, short row, short col)
{
   _settextposition(row, col);
   _outtext(buffer);
}

////////////////////////////////////////////////////////////////////////
//
//   Convert latitude in degrees to deg/min format
//
void ConvLatDegDM(double LatDeg, char *LatDM)
{   
   short    Deg, Min, MinFrac;
   char   Dir;
    
   if(LatDeg < 0){
      Dir = 'S'; 
      LatDeg = -LatDeg;
   }
   else{    
      Dir = 'N';
   }  
   Deg = (short) LatDeg; 
   LatDeg = (LatDeg - Deg) * 60.;
   Min = (short) LatDeg; 
   LatDeg = (LatDeg - Min) * 10000.;
   MinFrac = (short) LatDeg; 
   sprintf(LatDM," %c%2d%02d.%04d'", Dir, Deg, Min, MinFrac); 
}

////////////////////////////////////////////////////////////////////////
//
//   Convert longitude in degrees to deg/min format
//
void ConvLonDegDM(double LonDeg, char *LonDM)
{    
   short    Deg, Min, MinFrac;
   char   Dir;
    
   if(LonDeg < 0){
      Dir = 'W'; 
      LonDeg = -LonDeg;
   }
   else{    
      Dir = 'E';
   }  
   Deg = (short) LonDeg; 
   LonDeg = (LonDeg - Deg) * 60.;
   Min = (short) LonDeg; 
   LonDeg = (LonDeg - Min) * 10000.;
   MinFrac = (short) LonDeg; 
   sprintf(LonDM,"%c%3d%02d.%04d'", Dir, Deg, Min, MinFrac); 
}

///////////////////////////////////////////////////////////////////////////////
//
//   ne2ndcm - ECEF to ENU Direction Cosine Matrix
//
//   DESCRIPTION
//
//   Generates ECEF to ENU Direction Cosine Matrix

void e2ndcm (double lat, double lon, double cen[3][3]) {
   
  double slat, clat, slon, clon;
  
  slat = sin (lat);
  clat = cos (lat);
  slon = sin (lon);
  clon = cos (lon);
  
  cen[0][0] = -slon;
  cen[0][1] = clon;
  cen[0][2] = 0.0;
  
  cen[1][0] = -clon * slat;
  cen[1][1] = -slon * slat;
  cen[1][2] = clat;
  
  cen[2][0] = clon * clat;
  cen[2][1] = slon * clat;
  cen[2][2] = slat;


  return;
    
}

//------------------------------------------------------------------------------------------------------------
// GeoidToENU: returns the local-level coordinates of a Lat/Lon position about a 
//                     reference Lat/Lon Position
// 

void GeoidToENU(tGeoPos GeoPos, tGeoPos RefGeoPos, double ENU[3])
{
   double ENU_Cosines[3][3];
   double ECEFpos[3];
   double sin_lat, rh, rxy, lat, lon;
   
   
   // Produce the NEU Local Level direction cosines data
   // based on the reference position's Latitude and Longitude
   //
   // note: this is a spherical coordinate transformation not an ellipsoidal
   //       coordinate transformation, the resulting errors  due to the misalignment
   //       of the local normal will reach their maximums (about 10 deg) at about 
   //       45 deg of latitude
   //           
   //
   e2ndcm( RefGeoPos.lat * DegToRad , RefGeoPos.lon * DegToRad, ENU_Cosines );
   
   
   // Compute the ECEF Position of the point in question
   //
   lat = GeoPos.lat * DegToRad ;
   lon = GeoPos.lon * DegToRad;

   sin_lat = sin(lat);
   
   rh = radius * sqrt(1.0 - eccentricity_squared * sin_lat * sin_lat);
   
   rxy = (rh + GeoPos.alt) * cos (lat);
   
   ECEFpos[0] = rxy * cos (lon) + x_offset;
   ECEFpos[1] = rxy * sin (lon) + y_offset;
   ECEFpos[2] = sin_lat * (GeoPos.alt + omega_sqrd * rh) + z_offset; 
   
//   printf(" R= %7f, X = %7f,  Y = %7f,  Z=%7f\n",rh,ECEFpos[0],ECEFpos[1],ECEFpos[2]);
   
   // Compute the ENU coordinates of the point in question
   //
   
   ENU[0] = ENU_Cosines[0][0]*ECEFpos[0] + ENU_Cosines[0][1]*ECEFpos[1] + ENU_Cosines[0][2]*ECEFpos[2];
   ENU[1] = ENU_Cosines[1][0]*ECEFpos[0] + ENU_Cosines[1][1]*ECEFpos[1] + ENU_Cosines[1][2]*ECEFpos[2];
   ENU[2] = ENU_Cosines[2][0]*ECEFpos[0] + ENU_Cosines[2][1]*ECEFpos[1] + ENU_Cosines[2][2]*ECEFpos[2];
   ENU[2] = ENU[2]-rh;

   return;


}   
